import { connect, MqttClient, OnMessageCallback } from "mqtt";
import { mqttBrokerUrl } from "./config";

/**
 * MQTT Client Singleton
 * This class implements the Singleton pattern to ensure only one instance exists.
 * It provides methods to subscribe to topics, add message listeners, and close the connection.
 * @class MQTT
 * @singleton
 * @example
 * const mqtt_instance = MQTT.getInstance();
 * mqtt_instance.Subscribe("your/topic");
 * mqtt_instance.AddOnMessageListener((topic, message) => {
 *   console.log(`Received message on topic ${topic}:`, message.toString());
 * });
 * mqtt_instance.close(); // To close the connection when done
 */
export default class MQTT {
  static instance: MQTT | null = null;
  client: MqttClient | undefined;
  constructor() {
    if (MQTT.instance) {
      return MQTT.instance;
    }
    this.client = connect(mqttBrokerUrl);

    MQTT.instance = this;
  }

  /**
   * Returns the singleton MQTT instance, creating it if necessary.
   * @return {MQTT} The singleton MQTT instance.
   */
  static getInstance(): MQTT {
    if (!MQTT.instance) {
      MQTT.instance = new MQTT();
    }
    return MQTT.instance;
  }

  /**
   * Subscribes to a given MQTT topic.
   * @param {string} topic - The topic to subscribe to.
   * @returns {Promise<boolean>} A promise that resolves to true if the subscription was successful, false otherwise.
   */
  async Subscribe(topic: string): Promise<boolean> {
    return new Promise((resolve, reject) => {
      this.client?.subscribe(topic, (err) => {
        if (err) {
          console.error("MQTT: Subscription attempt error:", err);
          reject(err);
        } else {
          console.log(`MQTT: Successfully Subscribed to topic: ${topic}`);
          resolve(true);
        }
      });
    });
  }

  /**
   * Adds a listener for incoming MQTT messages.
   * @param {OnMessageCallback} callback
   */
  AddOnMessageListener(callback: OnMessageCallback): void {
    this.client?.on("message", callback);
  }

  /**
   * Closes the MQTT client connection.
   * @returns {void}
   */
  close(): void {
    this.client?.end(() => {
      console.log("MQTT client disconnected");
    });
  }
}
